module net.BurtonRadons.digc.program;

private import std.path;
private import std.string;

private import net.BurtonRadons.digc.digc;

extern (C)
char [] fmt (char [] format, ...)
{
    char [8192] buffer;
    int length;
    
    length = vsprintf (buffer, toStringz (format), cast (va_list) (&format + 1));
    return buffer [0 .. length].dup;
}

char [] indent (char [] text, int count)
{
    char [] space = new char [count + 1];
    
    space [] = ' ';
    space [0] = '\n';
    return space [1 .. space.length] ~ std.string.replace (text [0 .. text.length - 1], "\n", space) ~ text [text.length - 1 .. text.length];
}

class Program
{
    char type; /* 'l' = library, 'e' = executable, 'x' = example program. */
    char [] name; /* Output name. */
    char [] [] files; /* File list. */
    char [] [] args; /* Argument list. */
    bit shared; /* Indicate a shared library. */
    bit install; /* Install created files. */
    bit doNotLink = false; /* Avoid linking the program. */
    char [] basePath;
    
    /** Execute the program. */
    void run ()
    {
        if (type != 'e' && type != 'x')
            throw new Error ("Can only execute programs or examples.");
        system (toStringz (name));
    }
    
    /** Return the list of files created with this program. */
    char [] [] createdFileList ()
    {
        char [] [] list;
        
        if (type == 'e' || type == 'x')
        {
            if (install)
                list ~= (dmd_bin ~ "/" ~ name ~ ".exe");
            else
                list ~= (name ~ ".exe");
        }
        else if (type == 'l')
        {
            if (install)
                list ~= (dmd_lib ~ "/" ~ name ~ ".lib");
            else
                list ~= (name ~ ".lib");
            if (shared)
            {
                if (install)
                    list ~= (sharedLibraryInstallDir ~ name ~ ".dll");
                else
                    list ~= (name ~ ".dll");
            }
        }
        else assert (0);
            
        return list;
    }
        
    
    /** Delete the files created with this program. */
    void uninstall ()
    {
        char [] [] list = createdFileList ();
        
        for (int c; c < list.length; c ++)
            safeRemove (list [c]);
    }
    
    /** Return the earliest modification time for the created files or zero if it was never created. */
    ulong modificationTime ()
    {
        char [] [] list = createdFileList ();
        ulong time, compare;
        
        for (int c; c < list.length; c ++)
        {
            if ((compare = fileTime (list [c])) == 0)
                continue;
            if (!time || compare < time)
                time = compare;
        }
        
        return time;
    }
    
    /** Build the program.  Abort and return an error code if there was a problem, else return zero. */
    int build (char [] [] args, bit verbose, bit verboseDesc, bit dontdeletetempdir)
    {
        char [] [] files = this.files.dup;
        char [] [] fileModules = files.dup;
        bit shared = this.shared;
        char [] resultFile;
        int result;
        ulong date = modificationTime ();
        
        int call (char [] command)
        {
            if (verbose)
                printf ("%.*s\n\n", command);
            return system (toStringz (command));
        }
        
        printf ("\n------------------------------------------------------------\n");
        if (type == 'l')
            printf ("Library %.*s\n", name);
        else if (type == 'e')
            printf ("Program %.*s\n", name);
        else if (type == 'x')
            printf ("Example %.*s\n", name);
        
        args = args.dup ~ this.args;

        if (files.length == 0)
            return 0;
        
        for (int e; e < files.length; e ++)
        {
            fileModules [e] = getBaseName (replace (replace (files [e], "/", "."), "\\", "."));
            fileModules [e] = fileModules [e] [0 .. rfind (fileModules [e], '.')];
        }

        if (type == 'l')
            args ~= "-c";

        if (type == 'e' || type == 'x')
            args ~= " " ~ name ~ ".exe";

        int cfileslength = files.length;
        
        versions = null;
        
        for (int c; c < args.length; c ++)
        {
            char [] arg = args [c];

            if (arg.length <= 9 || arg [0 .. 9] != "-version=")
                continue;
            versions [arg [9 .. arg.length]] = true;
        }

        if (verboseDesc)
            printf ("Searching source for imports.\n");
        for (int c; c < cfileslength; c ++)
            searchFile (files [c], files, fileModules [c]);
            
        bit modified;
            
        for (int c; c < files.length; c ++)
        {
            ulong compare;
            
            if (endswith (files [c], ".lib"))
            {
                compare = fileTime (dmc_lib ~ "/" ~ files [c]);
                if (!compare)
                    compare = fileTime (dmd_lib ~ "/" ~ files [c]);
            }
            else
                compare = fileTime (files [c]);
                
            if (compare > date)
                modified = true;
        }
        
        if (!modified)
        {
            printf ("Files are not modified, skipping compilation.\n");
            return 0;
        }

        if (winMain)
            args ~= "-L/exet:nt/su:windows";

        char [] command;

        command ~= dmdexe;
        for (int c; c < args.length; c ++)
            command ~= " " ~ args [c];
        for (int c; c < files.length; c ++)
            command ~= " " ~ files [c];

        command ~= r" -I\dmd\src";

        if (type == 'e' || type == 'x')
        {
            byte [] data = (byte []) replace (resbase, "%EXE%", name);

            int point = 8;

            *(int *) &data [point] = (int) (data.length - (point + 4));
            write (resfilename, data);
            command ~= " " ~ resfilename;
        }

        sys.mkdir (tempDirectory);

        if (verboseDesc)
        {
            printf ("Expanding import libraries... ");
            fflush (std.c.stdio.stdout);
        }
        expandLibraries (files, tempDirectory);

        if (expandFileCount)
            printf ("%d files and %.2f kb expanded\n", expandFileCount, expandByteCount / 1024.0);
        else
            printf ("\n");

        command ~= " -I" ~ tempDirectory;

        result = call (command);

        if (!dontdeletetempdir)
            sys.rmdir(tempDirectory, true);

        if (type == 'e' || type == 'x')
            std.file.remove (resfilename);

        if (type == 'l' && !result)
        {
            char [] [] exports;

            if (shared)
            {
                /* Read in the exports. */
                for (int c; c < files.length; c ++)
                {
                    if (getExt (files [c]) == "d")
                    {
                        char [] name = std.path.getBaseName(files [c] [0 .. rfind (files [c], '.')] ~ ".obj");
                        char [] [] add;

                        add = omfListExports (name);
                        exports ~= add;
                    }
                }

                /* Create the DLL def file. */
                char [] text;

                text ~= "LIBRARY " ~ name ~ "\n"; /* This is currently ignored. */
                text ~= "DESCRIPTION \"Shared library created by digc\"\n";
                text ~= "EXETYPE NT\n";
                text ~= "CODE PRELOAD DISCARDABLE\n";
                text ~= "DATA PRELOAD SINGLE\n";
                text ~= "\n";
                text ~= "EXPORTS\n";
                for (int c; c < exports.length; c ++)
                {
                    char [] e = exports [c];

                    if (e [0] == '_')
                        e = e [1 .. e.length];
                    if (std.string.find (e, '@') != -1)
                        text ~= "\"" ~ e ~ "\"\n";
                    else
                        text ~= e ~ "\n";
                }

                std.file.write (deffilename, (byte []) text);

                /* Create the file containing DLLMain. */
                text = dllmainbase;

                std.file.write (sharedmainfilename, (byte []) text);

                char [] firstObject;

                /* Build "dmd objfiles deffile" and call it. */
                command = dmdexe;
                command ~= " " ~ deffilename;
                for (int c; c < files.length; c ++)
                {
                    if (getExt (files [c]) == "d")
                    {
                        char [] object = std.path.getBaseName (files [c] [0 .. rfind (files [c], '.')] ~ ".obj");

                        if (firstObject == null)
                            firstObject = object;
                        command ~= " " ~ object;
                    }
                    else if (getExt (files [c]) == "lib")
                        command ~= " " ~ files [c];
                }

                command ~= " snn.lib";

                command ~= " -L/IMPLIB:" ~ name ~ ".lib";

                command ~= " " ~ sharedmainfilename;

                result = call (command);

                /* Call "rename firstObject.dll name.dll" */
                if (!result && std.path.addExt (firstObject, "dll") != name ~ ".dll")
                {
                    safeRemove (name ~ ".dll");
                    call ("rename " ~ std.path.addExt (firstObject, "dll") ~ " " ~ name ~ ".dll"); 
                }

                safeRemove (std.path.addExt (firstObject, "map"));
                safeRemove (deffilename);
                safeRemove (sharedmainfilename);
                safeRemove (std.path.addExt (sharedmainfilename, "obj"));
            }
            else
            {
                command = libexe;
                command ~= " -c " ~ name ~ ".lib";
                for (int c; c < files.length; c ++)
                {
                    if (getExt (files [c]) == "d")
                        command ~= " " ~ std.path.getBaseName(files [c] [0 .. rfind (files [c], '.')] ~ ".obj");
                }

                result = call (command);
            }

            if (verboseDesc)
                printf ("Appending stripped import content to the library.\n");
            {
                File lib = new File (name ~ ".lib", FileMode.Out);
                uint [char []] fileOffsets;
                uint dirOffset;
                int depCount;

                lib.seekEnd (0);
                for (int c; c < files.length; c ++)
                {
                    if (getExt (files [c]) == "d")
                    {
                        char [] data = dstrip ((char[]) read (files [c]));

                        fileOffsets [files [c]] = lib.position ();
                        lib.write ((char []) data);
                    }
                    else if (getExt (files [c]) == "lib")
                        depCount ++;
                }

                dirOffset = lib.position ();
                lib.write (fileOffsets.keys.length);
                for (int c; c < files.length; c ++)
                {
                    if (getExt (files [c]) == "d")
                    {
                        lib.write (fileModules [c]);
                        lib.write (fileOffsets [files [c]]);
                    }
                }

                lib.write (depCount);
                for (int c; c < files.length; c ++)
                {
                    if (getExt (files [c]) == "lib")
                        lib.write (files [c]);
                }

                lib.write (dirOffset);
                lib.write (libraryMarker);
                delete lib;
            }
        }

        if (type == 'e' || type == 'x')
            safeRemove (std.path.addExt (dig_getBaseName (name), "map"));

        for (int c; c < files.length; c ++)
        {
            if (std.path.getExt (files [c]) == "d" && !doNotLink)
                safeRemove (std.path.addExt (dig_getBaseName (files [c]), "obj"));
        }

        if (result)
        {
            writeLibraries ();
            return result;
        }
        else if (install)
        {
            if (verboseDesc)
                printf ("Installing files.\n");
            if (type == 'e' || type == 'x')
                system (toStringz ("move /Y " ~ name ~ ".exe " ~ dmd_bin));
            else if (type == 'l')
            {
                printf ("    " ~ name ~ ".lib into " ~ dmd_lib ~ "\n");
                system (toStringz ("move /Y " ~ name ~ ".lib " ~ dmd_lib));
                if (shared)
                {
                    printf ("    " ~ name ~ ".dll into " ~ sharedLibraryInstallDir ~ "\n");
                    system (toStringz ("move /Y " ~ name ~ ".dll " ~ sharedLibraryInstallDir));
                }
            }
        }
        
        return 0;
    }
    
    char [] toString ()
    {
        char [] text;
        
        if (type == 'l')
            text ~= fmt ("library %.*s\n", name);
        else
            text ~= fmt ("program %.*s\n", name);
        text ~= "{\n";
        
        if (shared)
            text ~= "    shared\n";
        if (install)
            text ~= "    install\n";
        if (doNotLink)
            text ~= "    doNotLink\n";
        
        for (int c; c < files.length; c ++)
        {
            char [] file = files [c];
            
            if (file.length >= basePath.length && std.string.replace (file, "\\", "/") [0 .. basePath.length] == basePath)
                file = file [basePath.length + 1 .. file.length];
                
            text ~= "    add " ~ file ~ "\n";
        }
        
        text ~= "}\n";
        return text;
    }
}

class Package
{
    char [] source; /**< Package source, such as "mars.platform". */
    Program [] programs; /**< The programs within this package. */
    Package [] packages; /**< Packages within this package. */
    
    /** Try to find the named program in this package or a child, or null if it can't be found. */
    Program findProgram (char [] name)
    {
        Program result;
        
        for (int c; c < programs.length; c ++)
            if (programs [c].name == name)
                return programs [c];
            
        for (int c; c < packages.length; c ++)
            if ((result = packages [c].findProgram (name)) !== null)
                return result;
            
        return null;
    }
    
    int build (char [] [] aargs, bit verbose, bit verboseDesc)
    {
        int result;
        
        for (int c; c < programs.length; c ++)
        {
            if (programs [c].type == 'x')
                continue;
            if ((result = programs [c].build (aargs, verbose, verboseDesc, false)) != 0)
                return result;
        }
            
        for (int c; c < packages.length; c ++)
            if ((result = packages [c].build (aargs, verbose, verboseDesc)) != 0)
                return result;
            
        return 0;
    }
    
    char [] toString ()
    {
        char [] text;
        
        text = fmt ("package %.*s\n", source);
        text ~= "{\n";
        bit count = false;
        for (int c; c < programs.length; c ++, count = true)
        {
            if (count)
                text ~= "\n";
            text ~= indent (programs [c].toString (), 4) ~ "\n";
        }
        
        for (int c; c < packages.length; c ++, count = true)
        {
            if (count)
                text ~= "\n";
            text ~= indent (packages [c].toString (), 4) ~ "\n";
        }
        
        text ~= "}";
        return text;
    }
}

class PackageRead
{
    Package package; /**< Package being read. */
    Program program; /**< Program being read. */
    char [] [] lines; /**< Source lines. */
    int current; /**< Current line index. */
    bit [char []] versions;
    char [] filename;
    
    /** Read the package. */
    this (char [] source)
    {
        package = new Package ();
        package.source = source;
        filename = std.string.replace (source, ".", "/") ~ "/build";
        lines = std.string.splitlines ((char []) std.file.read (filename));
        
        versions ["MicrosoftWindows"] = false;
        version (Win32) versions ["MicrosoftWindows"] = true;

        while (1)
        {
            char [] [] words = line ();

            if (!words.length)
                break;
            exec (words);
        }
    }
    
    /** Read a line or return null if at the end. */
    char [] [] line ()
    {
        char [] [] words;
        
        do
        {
            if (current >= lines.length)
                return null;
            words = std.string.split (std.string.strip (lines [current ++]));
        }
        while (!words.length);
            
        return words;
    }
    
    void lcurlyMatch ()
    {
        int start = current;
        char [] [] words = line ();
        
        if (words.length != 1 || words [0] != "{")
            throw new Error (fmt ("Expected '{' after line %d in build file %.*s.", start, filename));
    }
    
    void exec (char [] [] words)
    {
        switch (words [0])
        {
            case "build":
                for (int d = 1; d < words.length; d ++)
                    package.packages ~= (new PackageRead (package.source ~ "." ~ words [d])).package;
                break;
            
            case "library":
                readProgram ('l', words [1]);
                break;
            
            case "example":
                readProgram ('x', words [1]);
                break;

            default:
                execBase (words, &exec);            
        }
    }
    
    void execBase (char [] [] words, void delegate (char [] []) func)
    {
        switch (words [0])
        {
            case "version":
                if (words.length != 2)
                    throw new Error (fmt ("'version' expected single argument at line %d in build file %.*s.", current, filename));
                if (!(words [1] in versions))
                    throw new Error (fmt ("Unknown 'version' value %.*s at line %d in build file %.*s.", words [1], current, filename));
                    
                if (versions [words [1]])
                    execBlock (func);
                else
                    skipBlock ();
                break;
            
            default:
                throw new Error (fmt ("Unknown build command '%.*s' at line %d in build file %.*s.", words [0], current, filename));
        }
    }
    
    void execProgram (char [] [] words)
    {
        if (words [0] [0] == '-')
        {
            for (int d = 0; d < words.length; d ++)
                program.args ~= words [d];
            return;
        }
        
        switch (words [0])
        {
            case "add":
                for (int d = 1; d < words.length; d ++)
                {
                    if (readWildcard (program, std.string.replace (package.source ~ "." ~ words [d], ".", "/") ~ ".d"))
                        throw new Error ("Could not find wildcard.");
                }
                
                break;
            
            case "addToArguments":
                for (int d = 1; d < words.length; d ++)
                    program.files ~= words [d];
                break;
            
            default:
                execBase (words, &execProgram);
        }
    }
    
    void readProgram (char type, char [] name)
    {
        program = new Program;
        package.programs ~= program;
        program.type = type;
        program.basePath = std.string.replace (package.source, ".", "/");
        
        if (type == 'l')
        {
            program.name = std.string.replace (package.source, ".", "_") ~ "_" ~ name;
            program.install = true;
        }
        else if (type == 'x')
            program.name = package.source ~ "." ~ name;
        else
            program.name = name;
        
        execBlock (&execProgram);
    }
    
    void execBlock (void delegate (char [] []) func)
    {
        lcurlyMatch ();
        
        while (1)
        {
            char [] [] words = line ();
            
            if (words [0] == "}")
                return;
            func (words);
        }
    }
    
    void skipBlock ()
    {
        int depth = 1;
        
        lcurlyMatch ();
        
        while (1)
        {
            char [] [] words = line ();
            
            if (words [0] == "{")
                depth ++;
            else if (words [0] == "}")
            {
                depth --;
                if (!depth)
                    return;
            }
        }
    }
}

Package readBuildFile (char [] filename)
{
    Package result = (new PackageRead (filename)).package;
    
    printf ("%.*s\n", result.toString ());
    return result;
}